//	TorusGames2DMaze.c
//
//	Maze2D coordinate conventions
//
//	Integer coordinates (h,v) label the cells, for h,v ∈ {0, 1, … , n-1}.
//	The first coordinate runs left-to-right, while the second runs bottom-to-top.
//	Cell centers align with the natural n×n grid, so for example
//	cells straddle the edges of the fundamental domain
//	and a single cell sits centered at the fundamental domain's corners.
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include "GeometryGamesSound.h"
#include <math.h>


#define NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_MAZE	5


//	How close must the user's tap come to the (field)mouse to count as a hit?
#define MOUSE_HIT_RADIUS	0.25

//	Compensate for the rotation in the raw graphics (value in radians, counterclockwise).
#define MOUSE_ROTATION		0.3


typedef struct
{
	unsigned int	itsH,
					itsV;
	Maze2DWallIndex	itsSide;
} WallSpec;

typedef struct
{
	unsigned int	itsH,
					itsV;
} CellSpec;

typedef struct
{
	CellSpec		itsSelf,
					itsParent;
} MazeTreeNode;


//	Public functions with private names
static void			MazeShutDown(ModelData *md);
static void			MazeReset(ModelData *md);
static bool			MazeDragBegin(ModelData *md, bool aRightClick);
static void			MazeDragObject(ModelData *md, double aHandLocalDeltaH, double aHandLocalDeltaV);
static void			MazeDragEnd(ModelData *md, double aDragDuration, bool aTouchSequenceWasCancelled);
static void			MazeSimulationUpdate(ModelData *md);

//	Private functions
static bool			AllocMazeCells(ModelData *md);
static void			FreeMazeCells(ModelData *md);
static bool			CreateNewMaze(ModelData *md);
static void			SetAllWallsTrue(ModelData *md);
static void			InitCellIndices(ModelData *md);
static void			InitWallList(ModelData *md, WallSpec *aWallList);
static void			RandomizeWallList(ModelData *md, WallSpec *aWallList);
static void			RemoveUnneededWalls(ModelData *md, WallSpec *aWallList);
static WallSpec		GetOtherSide(WallSpec aNearSide, unsigned int aMazeSize, TopologyType aTopology);
static void			PositionMouseAndCheese(ModelData *md);
static void			FindFurthestPoint(ModelData *md, CellSpec aNearCell, CellSpec *aFarCell);


void Maze2DSetUp(ModelData *md)
{
	//	Initialize function pointers.
	md->itsGameShutDown					= &MazeShutDown;
	md->itsGameReset					= &MazeReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= NULL;
	md->itsGame2DDragBegin				= &MazeDragBegin;
	md->itsGame2DDragObject				= &MazeDragObject;
	md->itsGame2DDragEnd				= &MazeDragEnd;
	md->itsGame3DDragBegin				= NULL;
	md->itsGame3DDragObject				= NULL;
	md->itsGame3DDragEnd				= NULL;
	md->itsGame3DGridSize				= NULL;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= &MazeSimulationUpdate;
	md->itsGameRefreshMessage			= NULL;

	//	Initialize variables.
	md->itsGameOf.Maze2D.itsSize		= 0;
	md->itsGameOf.Maze2D.itsMazeIsValid	= false;
	md->itsGameOf.Maze2D.itsCells		= NULL;
	
	//	Create a maze.
	MazeReset(md);
}


static void MazeShutDown(ModelData *md)
{
	//	Free the maze cells.
	FreeMazeCells(md);
}


static void MazeReset(ModelData *md)
{
	ErrorText	theErrorMessage	= NULL;

#if defined(GAME_CONTENT_FOR_SCREENSHOT) || defined(MAKE_GAME_CHOICE_ICONS)
#if 0	//	experiment with different seeds, to see which looks good?
	{
		static unsigned int	theSeed	= 0;

		RandomInitWithSeed(theSeed);
		printf("using 2D maze seed %d\n", theSeed);
		theSeed++;
	}
#else	//	use our favorite seed
	{
		//	Caution:  The random number generator's implementation is different
		//	on different platforms, so while a given seed generates the same maze
		//	on iOS and macOS, we'd need to use a different seed on other platforms.
#ifdef GAME_CONTENT_FOR_SCREENSHOT
		RandomInitWithSeed(8);
#endif
#ifdef MAKE_GAME_CHOICE_ICONS
		RandomInitWithSeed(93);
#endif
	}
#endif	//	experiment or use favorite seed
#endif	//	defined(GAME_CONTENT_FOR_SCREENSHOT) || defined(MAKE_GAME_CHOICE_ICONS)

	FreeMazeCells(md);

	switch (md->itsDifficultyLevel)
	{
		case 0:  md->itsGameOf.Maze2D.itsSize =  4;  break;
		case 1:  md->itsGameOf.Maze2D.itsSize =  8;  break;
		case 2:  md->itsGameOf.Maze2D.itsSize = 16;  break;
		case 3:  md->itsGameOf.Maze2D.itsSize = 32;  break;
		default: md->itsGameOf.Maze2D.itsSize =  2;  break;	//	should never occur
	}

	if ( ! AllocMazeCells(md) )
	{
		theErrorMessage = u"Couldn't allocate memory for maze walls.";
		goto CleanUpMazeReset;
	}

	if ( ! CreateNewMaze(md) )
	{
		theErrorMessage = u"Couldn't allocate memory to create new maze.";
		goto CleanUpMazeReset;
	}

	//	Place the mouse and the cheese as far from each other as possible.
	PositionMouseAndCheese(md);

#ifdef GAME_CONTENT_FOR_SCREENSHOT
	//	Tweak the mouse's position to seem a little more natural.
	md->itsGameOf.Maze2D.itsMousePlacement.itsH		-= 0.7500 / (double)md->itsGameOf.Maze2D.itsSize;
	md->itsGameOf.Maze2D.itsMousePlacement.itsV		+= 0.0625 / (double)md->itsGameOf.Maze2D.itsSize;
	md->itsGameOf.Maze2D.itsMousePlacement.itsAngle	-= PI / 6.0;
#endif
#ifdef MAKE_GAME_CHOICE_ICONS
	md->itsGameOf.Maze2D.itsMousePlacement.itsFlip = true;
#endif

	//	Abort any pending simulation.
	SimulationEnd(md);

	//	Ready to go.
	md->itsGameIsOver	= false;
	md->itsFlashFlag	= false;

CleanUpMazeReset:

	if (theErrorMessage != NULL)
	{
		md->itsGameOf.Maze2D.itsMazeIsValid = false;
		GeometryGamesErrorMessage(theErrorMessage, u"MazeReset() Error");
	}
	else
	{
		md->itsGameOf.Maze2D.itsMazeIsValid = true;
	}
}


static bool AllocMazeCells(ModelData *md)
{
	unsigned int	i;

	md->itsGameOf.Maze2D.itsCells = (Maze2DCell **) GET_MEMORY(md->itsGameOf.Maze2D.itsSize * sizeof(Maze2DCell *));
	if (md->itsGameOf.Maze2D.itsCells == NULL)
		return false;

	for (i = 0; i < md->itsGameOf.Maze2D.itsSize; i++)
	{
		md->itsGameOf.Maze2D.itsCells[i] = (Maze2DCell *) GET_MEMORY(md->itsGameOf.Maze2D.itsSize * sizeof(Maze2DCell));
		if (md->itsGameOf.Maze2D.itsCells[i] == NULL)
		{
			while (i-- > 0)
				FREE_MEMORY(md->itsGameOf.Maze2D.itsCells[i]);
			FREE_MEMORY(md->itsGameOf.Maze2D.itsCells);
			md->itsGameOf.Maze2D.itsCells = NULL;
			return false;
		}
	}

	return true;
}


static void FreeMazeCells(ModelData *md)
{
	unsigned int	i;

	if (md->itsGameOf.Maze2D.itsCells != NULL)
	{
		for (i = 0; i < md->itsGameOf.Maze2D.itsSize; i++)
			FREE_MEMORY(md->itsGameOf.Maze2D.itsCells[i]);

		FREE_MEMORY(md->itsGameOf.Maze2D.itsCells);

		md->itsGameOf.Maze2D.itsCells = NULL;
	}
}


static bool CreateNewMaze(ModelData *md)
{
	//	This algorithm is from the Maze program written
	//	by Jeff Weeks for Adam Weeks Marano, Christmas 1992.

	//	The plan is to give each cell its own index.  We then put
	//	all the possible Walls (WallWest and WallSouth for each cell)
	//	on a list, randomize the order of the list, and then go down it
	//	one item at a time.  Whenever a wall separates two cells
	//	with different indices, erase it and merge the neighboring indices
	//	(that is, set all occurrences of one index to equal the value
	//	of the other index).  When a wall separates two cells
	//	of the same index, leave it there.  It's easy to see that
	//	this will yield a maze with a unique path between any two points.

	//	We assume memory for md->itsGameOf.Maze2D.itsCells has already been allocated.

	WallSpec	*theWallList	= NULL;

	SetAllWallsTrue(md);
	InitCellIndices(md);
	theWallList = (WallSpec *) GET_MEMORY(2 * md->itsGameOf.Maze2D.itsSize * md->itsGameOf.Maze2D.itsSize * sizeof(WallSpec));
	if (theWallList == NULL)
		return false;
	InitWallList(md, theWallList);
	RandomizeWallList(md, theWallList);
	RemoveUnneededWalls(md, theWallList);
	FREE_MEMORY_SAFELY(theWallList);

	return true;
}


static void SetAllWallsTrue(ModelData *md)
{
	unsigned int	i,
					j,
					k;

	for (i = 0; i < md->itsGameOf.Maze2D.itsSize; i++)
		for (j = 0; j < md->itsGameOf.Maze2D.itsSize; j++)
			for (k = 0; k < 4; k++)
				md->itsGameOf.Maze2D.itsCells[i][j].itsWalls[k] = true;
}


static void InitCellIndices(ModelData *md)
{
	unsigned int	theCount,
					i,
					j;

	theCount = 0;

	for (i = 0; i < md->itsGameOf.Maze2D.itsSize; i++)
		for (j = 0; j < md->itsGameOf.Maze2D.itsSize; j++)
			md->itsGameOf.Maze2D.itsCells[i][j].itsIndex = theCount++;
}


static void InitWallList(
	ModelData	*md,
	WallSpec	*aWallList)
{
	WallSpec		theWallSpec;
	unsigned int	theCount;

	theCount = 0;

	for (theWallSpec.itsH = 0; theWallSpec.itsH < md->itsGameOf.Maze2D.itsSize; theWallSpec.itsH++)
		for (theWallSpec.itsV = 0; theWallSpec.itsV < md->itsGameOf.Maze2D.itsSize; theWallSpec.itsV++)
			for (theWallSpec.itsSide = 0; theWallSpec.itsSide < 2; theWallSpec.itsSide++)	//	WallWest and WallSouth
				aWallList[theCount++] = theWallSpec;
}


static void RandomizeWallList(
	ModelData	*md,
	WallSpec	*aWallList)
{
	unsigned int	theNumWallsRemaining,
					theLastIndex,
					theRandomIndex;
	WallSpec		theTempWallSpec;

	theNumWallsRemaining = 2 * md->itsGameOf.Maze2D.itsSize * md->itsGameOf.Maze2D.itsSize;

	while (theNumWallsRemaining >= 2)
	{
		
		theRandomIndex	= RandomUnsignedInteger() % theNumWallsRemaining;
		theLastIndex	= theNumWallsRemaining - 1;

		theTempWallSpec				= aWallList[theLastIndex];
		aWallList[theLastIndex]		= aWallList[theRandomIndex];
		aWallList[theRandomIndex]	= theTempWallSpec;
		
		theNumWallsRemaining--;
	}
}


static void RemoveUnneededWalls(
	ModelData	*md,
	WallSpec	*aWallList)
{
	WallSpec		theNearSide,
					theFarSide;
	unsigned int	theNumWalls,
					theNearIndex,
					theFarIndex,
					i,
					h,
					v;

	theNumWalls = 2 * md->itsGameOf.Maze2D.itsSize * md->itsGameOf.Maze2D.itsSize;

	for (i = 0; i < theNumWalls; i++)
	{
		theNearSide	= aWallList[i];
		theFarSide	= GetOtherSide(theNearSide, md->itsGameOf.Maze2D.itsSize, md->itsTopology);

		theNearIndex = md->itsGameOf.Maze2D.itsCells[theNearSide.itsH][theNearSide.itsV].itsIndex;
		theFarIndex  = md->itsGameOf.Maze2D.itsCells[ theFarSide.itsH][ theFarSide.itsV].itsIndex;

		if (theNearIndex != theFarIndex)
		{
			for (h = 0; h < md->itsGameOf.Maze2D.itsSize; h++)
				for (v = 0; v < md->itsGameOf.Maze2D.itsSize; v++)
					if (md->itsGameOf.Maze2D.itsCells[h][v].itsIndex == theNearIndex)
						md->itsGameOf.Maze2D.itsCells[h][v].itsIndex = theFarIndex;

			md->itsGameOf.Maze2D.itsCells[theNearSide.itsH][theNearSide.itsV].itsWalls[theNearSide.itsSide] = false;
			md->itsGameOf.Maze2D.itsCells[ theFarSide.itsH][ theFarSide.itsV].itsWalls[ theFarSide.itsSide] = false;
		}
	}
}


static WallSpec GetOtherSide(
	WallSpec		aNearSide,
	unsigned int	aMazeSize,
	TopologyType	aTopology)
{
	WallSpec	theFarSide;

	switch (aNearSide.itsSide)
	{
		case WallWest:
			if (aNearSide.itsH > 0)
				theFarSide.itsH	= aNearSide.itsH - 1;
			else
				theFarSide.itsH	= aMazeSize - 1;
			theFarSide.itsV		= aNearSide.itsV;
			theFarSide.itsSide	= WallEast;
			break;

		case WallSouth:
			theFarSide.itsH		= aNearSide.itsH;
			if (aNearSide.itsV > 0)
				theFarSide.itsV	= aNearSide.itsV - 1;
			else
			{
				theFarSide.itsV = aMazeSize - 1;
				if (aTopology == Topology2DKlein)
					theFarSide.itsH = (theFarSide.itsH > 0) ? aMazeSize - theFarSide.itsH : 0;
			}
			theFarSide.itsSide	= WallNorth;
			break;

		case WallEast:
			if (aNearSide.itsH < aMazeSize - 1)
				theFarSide.itsH	= aNearSide.itsH + 1;
			else
				theFarSide.itsH	= 0;
			theFarSide.itsV		= aNearSide.itsV;
			theFarSide.itsSide	= WallWest;
			break;

		case WallNorth:
			theFarSide.itsH		= aNearSide.itsH;
			if (aNearSide.itsV < aMazeSize - 1)
				theFarSide.itsV	= aNearSide.itsV + 1;
			else
			{
				theFarSide.itsV = 0;
				if (aTopology == Topology2DKlein)
					theFarSide.itsH = (theFarSide.itsH > 0) ? aMazeSize - theFarSide.itsH : 0;
			}
			theFarSide.itsSide	= WallSouth;
			break;
		
		default:	//	unused, suppresses compiler warnings
			theFarSide.itsH		= 0;
			theFarSide.itsV		= 0;
			theFarSide.itsSide	= WallWest;
			break;
	}

	return theFarSide;
}


static void PositionMouseAndCheese(ModelData *md)
{
	//	Proposition
	//
	//	Let cell A be an arbitrary cell in the maze.
	//	Let cell B be a cell maximally far from cell A
	//		(following the maze, of course).
	//	Let cell C be a cell maximally far from cell B
	//		(following the maze, of course).
	//	Then the distance from cell B to cell C is
	//		greater than or equal to the distance between
	//		any other pair of cells.
	//
	//	The proof is easy, but not quite so brief that
	//	I feel like writing it down here.  Presumably this
	//	result appears somewhere in the literature, even
	//	though I couldn't find it on the web (and Patti Lock
	//	was unaware of it).

	CellSpec		theCellA = {0, 0},
					theCellB,
					theCellC;
	unsigned int	theSide;

	//	Find a pair of maximally distant points in the maze.
	FindFurthestPoint(md, theCellA, &theCellB);
	FindFurthestPoint(md, theCellB, &theCellC);

	//	Place the mouse, converting from integer "maze coordinates" in [0, size)
	//	to floating-point "fundamental domain coordinates" in [-0.5, +0.5).
	//	Scale the mouse by 7/8 = 0.875 to avoid overlapping the maze walls.
	md->itsGameOf.Maze2D.itsMousePlacement.itsH		= (double)( (signed int)(2 * theCellB.itsH) - (signed int)(md->itsGameOf.Maze2D.itsSize) )
													/ (double)( 2 * md->itsGameOf.Maze2D.itsSize );
	md->itsGameOf.Maze2D.itsMousePlacement.itsV		= (double)( (signed int)(2 * theCellB.itsV) - (signed int)(md->itsGameOf.Maze2D.itsSize) )
													/ (double)( 2 * md->itsGameOf.Maze2D.itsSize );
	md->itsGameOf.Maze2D.itsMousePlacement.itsFlip	= false;
	md->itsGameOf.Maze2D.itsMousePlacement.itsAngle	= 0.0;	//	to be overwritten immediately below, except in 1×1 case
	md->itsGameOf.Maze2D.itsMousePlacement.itsSizeH	= 0.875 * (1.0 / md->itsGameOf.Maze2D.itsSize);
	md->itsGameOf.Maze2D.itsMousePlacement.itsSizeV	= 0.875 * (1.0 / md->itsGameOf.Maze2D.itsSize);

	//	Set the mouse's heading, measured in radians counterclockwise from east.
	for (theSide = 0; theSide < 4; theSide++)	//	theSide = WallWest, WallSouth, WallEast, WallNorth
		if ( ! md->itsGameOf.Maze2D.itsCells[theCellB.itsH][theCellB.itsV].itsWalls[theSide] )
			md->itsGameOf.Maze2D.itsMousePlacement.itsAngle = -PI + (0.5*PI)*theSide + MOUSE_ROTATION;

	//	Place the cheese, converting from integer "maze coordinates" in [0, size)
	//	to floating-point "fundamental domain coordinates" in [-0.5, +0.5).
	//	Scale the cheese by a little less than 7/8 = 0.875 to avoid touching the maze walls.
	md->itsGameOf.Maze2D.itsCheesePlacement.itsH		= (double)( (signed int)(2 * theCellC.itsH) - (signed int)(md->itsGameOf.Maze2D.itsSize) )
														/ (double)( 2 * md->itsGameOf.Maze2D.itsSize );
	md->itsGameOf.Maze2D.itsCheesePlacement.itsV		= (double)( (signed int)(2 * theCellC.itsV) - (signed int)(md->itsGameOf.Maze2D.itsSize) )
														/ (double)( 2 * md->itsGameOf.Maze2D.itsSize );
	md->itsGameOf.Maze2D.itsCheesePlacement.itsFlip		= false;
	md->itsGameOf.Maze2D.itsCheesePlacement.itsAngle	= 0.0;
	md->itsGameOf.Maze2D.itsCheesePlacement.itsSizeH	= 0.8 * (1.0 / md->itsGameOf.Maze2D.itsSize);
	md->itsGameOf.Maze2D.itsCheesePlacement.itsSizeV	= 0.8 * (1.0 / md->itsGameOf.Maze2D.itsSize);
}


static void FindFurthestPoint(
	ModelData	*md,
	CellSpec	aNearCell,
	CellSpec	*aFarCell)
{
	MazeTreeNode	*theQueue		= NULL;
	CellSpec		theInvalidCell	= {-1, -1};
	unsigned int	theQueueStart,
					theQueueEnd;
	CellSpec		theSelf,
					theParent;
	unsigned int	theSide;
	WallSpec		theSelfWallSpec,
					theNeighborWallSpec;

	//	md->itsGameOf.Maze2D.itsSize will never be 0,
	//	but somehow the static analyzer doesn't grasp that.
	//	So let's check it explicitly, to suppress
	//	the "zero-allocated memory" warning.
	if (md->itsGameOf.Maze2D.itsSize == 0)
	{
		GeometryGamesErrorMessage(	u"Impossible condition #1 in FindFurthestPoint().",
						u"Internal Error");
		exit(1);
	}

	theQueue = (MazeTreeNode *) GET_MEMORY(md->itsGameOf.Maze2D.itsSize * md->itsGameOf.Maze2D.itsSize * sizeof(MazeTreeNode));
	if (theQueue == NULL)
	{
		GeometryGamesErrorMessage(	u"Couldn't get memory to properly position mouse and cheese in maze.",
						u"FindFurthestPoint() Error");
		aFarCell->itsH = (md->itsGameOf.Maze2D.itsSize > 1) ? ( ! aNearCell.itsH) : 0;
		aFarCell->itsV = 0;
		return;
	}

	theQueueStart			= 0;
	theQueueEnd				= 0;
	theQueue[0].itsSelf		= aNearCell;
	theQueue[0].itsParent	= theInvalidCell;

	while (theQueueStart <= theQueueEnd)
	{
		//	Consider the next node on the queue.
		theSelf		= theQueue[theQueueStart].itsSelf;
		theParent	= theQueue[theQueueStart].itsParent;

		//	Each of theNode's accessible neighbors,
		//	excluding its own parent, gets added to the queue.
		for (theSide = 0; theSide < 4; theSide++)	//	theSide = WallWest, WallSouth, WallEast, WallNorth
			if ( ! md->itsGameOf.Maze2D.itsCells[theSelf.itsH][theSelf.itsV].itsWalls[theSide] )
			{
				theSelfWallSpec.itsH	= theSelf.itsH;
				theSelfWallSpec.itsV	= theSelf.itsV;
				theSelfWallSpec.itsSide	= theSide;

				theNeighborWallSpec = GetOtherSide(theSelfWallSpec, md->itsGameOf.Maze2D.itsSize, md->itsTopology);

				if (theNeighborWallSpec.itsH == theParent.itsH
				 && theNeighborWallSpec.itsV == theParent.itsV)
					continue;

				theQueueEnd++;
				theQueue[theQueueEnd].itsSelf.itsH	= theNeighborWallSpec.itsH;
				theQueue[theQueueEnd].itsSelf.itsV	= theNeighborWallSpec.itsV;
				theQueue[theQueueEnd].itsParent		= theSelf;
			}

		theQueueStart++;
	}

	if (theQueueStart != md->itsGameOf.Maze2D.itsSize * md->itsGameOf.Maze2D.itsSize)
	{
		GeometryGamesErrorMessage(	u"Impossible condition #2 in FindFurthestPoint().",
						u"Internal Error");
		exit(1);
	}

	aFarCell->itsH = theSelf.itsH;
	aFarCell->itsV = theSelf.itsV;

	FREE_MEMORY_SAFELY(theQueue);
}


static bool MazeDragBegin(
	ModelData	*md,
	bool		aRightClick)
{
	unsigned int	i,
					j;

	UNUSED_PARAMETER(aRightClick);

	if ( ! md->itsGameOf.Maze2D.itsMazeIsValid )
		return false;

	//	Did we hit the (field)mouse?
	//
	//	Note:  In a "difficult" or "extra difficult" maze,
	//	the (field)mouse will be tiny.  There's no need
	//	to insist that the user hit it exactly.
	//	Even touching a nearby cell is close enough.
	//
	md->its2DDragIsReversed = (	md->its2DHandPlacement.itsFlip
							 != md->itsGameOf.Maze2D.itsMousePlacement.itsFlip);
	if (Shortest2DDistance(
							md->its2DHandPlacement.itsH,
							md->its2DHandPlacement.itsV,
							md->itsGameOf.Maze2D.itsMousePlacement.itsH,
							md->itsGameOf.Maze2D.itsMousePlacement.itsV,
							md->itsTopology,
							&md->its2DDragIsReversed)
		< MOUSE_HIT_RADIUS)
	{
		//	Clear the history.
		for (i = 0; i < DRAG_HISTORY_DEPTH; i++)
			for (j = 0; j < 2; j++)
				md->its2DDragHistory[i][j] = 0.0;

		//	Report a hit.
		return true;
	}
	else
	{
		return false;
	}
}


static void MazeDragObject(
	ModelData	*md,
	double		aHandLocalDeltaH,
	double		aHandLocalDeltaV)
{
	double			theMouseLocalDeltaH,
					theMouseLocalDeltaV,
					theHandAmbientDeltaH,
					theHandAmbientDeltaV,
					theMouseAmbientDeltaH,
					theMouseAmbientDeltaV;
	unsigned int	theCellH,
					theCellV,
					theNormalizedCellH,
					theNormalizedCellV;
	double			theCellCenterH,
					theCellCenterV,
					theRelativePositionH,
					theRelativePositionV,
					theNormalizedRelativePositionH,
					theNormalizedRelativePositionV,
					theCellWidth;
	signed int		theSignH,
					theSignV;
	double			theSizeH,
					theSizeV,
					theRadius;
	unsigned int	i,
					j;
	double			theTotalMouseLocalDeltaH,
					theTotalMouseLocalDeltaV;

	if ( ! md->itsGameOf.Maze2D.itsMazeIsValid )
		return;

	//	Compute the requested motion relative to the (field)mouse.

	theMouseLocalDeltaH		= md->its2DDragIsReversed
							? -aHandLocalDeltaH : aHandLocalDeltaH;
	theMouseLocalDeltaV		= aHandLocalDeltaV;

	theMouseAmbientDeltaH	= md->itsGameOf.Maze2D.itsMousePlacement.itsFlip
							? -theMouseLocalDeltaH : theMouseLocalDeltaH;
	theMouseAmbientDeltaV	= theMouseLocalDeltaV;

	//	Which cell are we in?
	//	Allow theCellH == md->itsGameOf.Maze2D.itsSize even though
	//	that's equivalent to theCellH == 0, and similarly for theCellV.
	//	We'll compute the normalized indices immediately below.
	theCellH = (unsigned int) floor((md->itsGameOf.Maze2D.itsMousePlacement.itsH + 0.5)*md->itsGameOf.Maze2D.itsSize + 0.5);
	theCellV = (unsigned int) floor((md->itsGameOf.Maze2D.itsMousePlacement.itsV + 0.5)*md->itsGameOf.Maze2D.itsSize + 0.5);

	//	Normalize theCellH and theCellV.
	theNormalizedCellH = theCellH;
	theNormalizedCellV = theCellV;
	if (theNormalizedCellV >= md->itsGameOf.Maze2D.itsSize)	//	special case for  top  edge of fundamental domain (plus protection from invalid input)
	{
		theNormalizedCellV = 0;
		if (md->itsTopology == Topology2DKlein)
			theNormalizedCellH = md->itsGameOf.Maze2D.itsSize - theNormalizedCellH;
	}
	if (theNormalizedCellH >= md->itsGameOf.Maze2D.itsSize)	//	special case for right edge of fundamental domain (plus protection from invalid input)
		theNormalizedCellH = 0;

	//	Where is the cell's center?
	//	(We're back to the non-normalized form here.)
	theCellCenterH = (double)theCellH/(double)md->itsGameOf.Maze2D.itsSize - 0.5;
	theCellCenterV = (double)theCellV/(double)md->itsGameOf.Maze2D.itsSize - 0.5;

	//	Express the new (field)mouse position relative to the cell's center.
	theRelativePositionH = md->itsGameOf.Maze2D.itsMousePlacement.itsH + theMouseAmbientDeltaH - theCellCenterH;
	theRelativePositionV = md->itsGameOf.Maze2D.itsMousePlacement.itsV + theMouseAmbientDeltaV - theCellCenterV;

	//	theRelativePositionV never depends on the normalization,
	//	but in a Klein bottle theRelativePositionH does.
	theNormalizedRelativePositionH =
		(md->itsTopology == Topology2DKlein && theCellV == md->itsGameOf.Maze2D.itsSize) ?
		-theRelativePositionH : theRelativePositionH;
	theNormalizedRelativePositionV = theRelativePositionV;

	//	Don't pass through any walls, or even go near them.
	if (md->itsGameOf.Maze2D.itsCells[theNormalizedCellH][theNormalizedCellV].itsWalls[WallWest ]
	 && theNormalizedRelativePositionH < 0)
		theNormalizedRelativePositionH = 0;
	if (md->itsGameOf.Maze2D.itsCells[theNormalizedCellH][theNormalizedCellV].itsWalls[WallSouth]
	 && theNormalizedRelativePositionV < 0)
		theNormalizedRelativePositionV = 0;
	if (md->itsGameOf.Maze2D.itsCells[theNormalizedCellH][theNormalizedCellV].itsWalls[WallEast ]
	 && theNormalizedRelativePositionH > 0)
		theNormalizedRelativePositionH = 0;
	if (md->itsGameOf.Maze2D.itsCells[theNormalizedCellH][theNormalizedCellV].itsWalls[WallNorth]
	 && theNormalizedRelativePositionV > 0)
		theNormalizedRelativePositionV = 0;
	
	//	Recover the unnormalized relative position.
	theRelativePositionH =
		(md->itsTopology == Topology2DKlein && theCellV == md->itsGameOf.Maze2D.itsSize) ?
		-theNormalizedRelativePositionH : theNormalizedRelativePositionH;
	theRelativePositionV = theNormalizedRelativePositionV;

	//	For convenience, note the cell size.
	theCellWidth = 1.0 / md->itsGameOf.Maze2D.itsSize;

	//	If the (field)mouse is moving outside the current cell...
	if (fabs(theRelativePositionH) >= 0.5*theCellWidth
	 || fabs(theRelativePositionV) >= 0.5*theCellWidth)
	{
		//	...snap to the nearest passageway's centerline
		if (fabs(theRelativePositionH) < fabs(theRelativePositionV))
			theRelativePositionH = 0;
		else
			theRelativePositionV = 0;

		//	..and don't travel any further than the center
		//	of the neighboring cell.
		if (theRelativePositionH >  theCellWidth)
			theRelativePositionH =  theCellWidth;
		if (theRelativePositionH < -theCellWidth)
			theRelativePositionH = -theCellWidth;
		if (theRelativePositionV >  theCellWidth)
			theRelativePositionV =  theCellWidth;
		if (theRelativePositionV < -theCellWidth)
			theRelativePositionV = -theCellWidth;
	}
	else
	{
		//	Otherwise the (field)mouse is staying within the current cell,
		//	and it suffices to stay a half cell-width away from
		//	corners (keeping in mind that we're already a half
		//	cell-width away from all walls -- see above).
		//	At this point you might want to make yourself
		//	a little sketch showing the cell, with a line
		//	through the center parallel to each wall,
		//	and a quarter-circle of radius one-half cell width
		//	centered at each vertex (or, more to the point,
		//	at each vertex not adjacent to a cell wall).

		//	To avoid special cases, split theRelativePosition
		//	into signs and magnitudes.
		theSignH = (theRelativePositionH >= 0) ? +1 : -1;
		theSignV = (theRelativePositionV >= 0) ? +1 : -1;
		theSizeH = fabs(theRelativePositionH);
		theSizeV = fabs(theRelativePositionV);

		//	Re-express the point relative to the nearest corner.
		theSizeH = 0.5*theCellWidth - theSizeH;
		theSizeV = 0.5*theCellWidth - theSizeV;

		//	Are we closer to the corner than a half a cell width?
		theRadius = sqrt(theSizeH*theSizeH + theSizeV*theSizeV);
		if (theRadius < 0.0001*theCellWidth)	//	Guard against unexpected input.
		{
			theRelativePositionH = 0.0;
			theRelativePositionV = 0.0;
		}
		else
		if (theRadius < 0.5*theCellWidth)	//	If we're too close to the corner...
		{
			//	...dilate radially to lie exactly
			//	a half cell width from the corner
			theSizeH *= 0.5*theCellWidth / theRadius;
			theSizeV *= 0.5*theCellWidth / theRadius;

			//	...switch back to cell-center coordinates
			theSizeH = 0.5*theCellWidth - theSizeH;
			theSizeV = 0.5*theCellWidth - theSizeV;

			//	...and multiply the signs back in
			//	to recover the adjusted RelativePosition.
			theRelativePositionH = theSignH * theSizeH;
			theRelativePositionV = theSignV * theSizeV;
		}
	}

	//	Now that we've clipped theRelativePosition to avoid
	//	the maze walls and corners, (re)compute the deltas.

	theMouseAmbientDeltaH	= theCellCenterH + theRelativePositionH - md->itsGameOf.Maze2D.itsMousePlacement.itsH;
	theMouseAmbientDeltaV	= theCellCenterV + theRelativePositionV - md->itsGameOf.Maze2D.itsMousePlacement.itsV;

	theMouseLocalDeltaH		= md->itsGameOf.Maze2D.itsMousePlacement.itsFlip
							? -theMouseAmbientDeltaH : theMouseAmbientDeltaH;
	theMouseLocalDeltaV		= theMouseAmbientDeltaV;

	aHandLocalDeltaH		= md->its2DDragIsReversed
							? -theMouseLocalDeltaH : theMouseLocalDeltaH;
	aHandLocalDeltaV		= theMouseLocalDeltaV;

	theHandAmbientDeltaH	= md->its2DHandPlacement.itsFlip
							? -aHandLocalDeltaH : aHandLocalDeltaH;
	theHandAmbientDeltaV	= aHandLocalDeltaV;

	//	Move the (field)mouse.
	md->itsGameOf.Maze2D.itsMousePlacement.itsH += theMouseAmbientDeltaH;
	md->itsGameOf.Maze2D.itsMousePlacement.itsV += theMouseAmbientDeltaV;
	Normalize2DPlacement(&md->itsGameOf.Maze2D.itsMousePlacement, md->itsTopology);

	//	Move the hand.
	md->its2DHandPlacement.itsH += theHandAmbientDeltaH;
	md->its2DHandPlacement.itsV += theHandAmbientDeltaV;
	Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);

	//	Update the drag history.
	for (i = DRAG_HISTORY_DEPTH - 1; i-- > 0; )
		for (j = 0; j < 2; j++)
			md->its2DDragHistory[i+1][j] = md->its2DDragHistory[i][j];
	md->its2DDragHistory[0][0] = theMouseLocalDeltaH;
	md->its2DDragHistory[0][1] = theMouseLocalDeltaV;

	//	Adjust the (field)mouse's heading.
	//	To avoid processor-speed dependencies, sum the (field)mouse's local motions
	//	over a fixed distance rather than a fixed number of frames.
	theTotalMouseLocalDeltaH = 0.0;
	theTotalMouseLocalDeltaV = 0.0;
	for (i = 0; i < DRAG_HISTORY_DEPTH; i++)
	{
		theTotalMouseLocalDeltaH += md->its2DDragHistory[i][0];
		theTotalMouseLocalDeltaV += md->its2DDragHistory[i][1];

		if (fabs(theTotalMouseLocalDeltaH) > 0.5 * theCellWidth
		 || fabs(theTotalMouseLocalDeltaV) > 0.5 * theCellWidth)
			break;
	}
	if (theTotalMouseLocalDeltaH != 0.0
	 || theTotalMouseLocalDeltaV != 0.0)
		md->itsGameOf.Maze2D.itsMousePlacement.itsAngle = atan2(theTotalMouseLocalDeltaV, theTotalMouseLocalDeltaH) + MOUSE_ROTATION;
}


static void MazeDragEnd(
	ModelData	*md,
	double		aDragDuration,	//	in seconds
	bool		aTouchSequenceWasCancelled)
{
	UNUSED_PARAMETER(aDragDuration);

	//	If a touch sequence got cancelled (perhaps because a gesture was recognized),
	//	return without testing for a win.
	if (aTouchSequenceWasCancelled)
		return;

	//	Did the mouse reach the cheese?

	if ( ! md->itsGameIsOver
	 && Shortest2DDistance(	md->itsGameOf.Maze2D.itsMousePlacement.itsH,
							md->itsGameOf.Maze2D.itsMousePlacement.itsV,
							md->itsGameOf.Maze2D.itsCheesePlacement.itsH,
							md->itsGameOf.Maze2D.itsCheesePlacement.itsV,
							md->itsTopology,
							NULL)
		< 0.25 / md->itsGameOf.Maze2D.itsSize )
	{
		md->itsGameIsOver = true;
		EnqueueSoundRequest(u"MazeMunch.wav");
		SimulationBegin(md, Simulation2DMazeFlash);
	}
}


static void MazeSimulationUpdate(ModelData *md)
{
	switch (md->itsSimulationStatus)
	{
		case Simulation2DMazeFlash:
			md->itsFlashFlag = (((unsigned int) floor(10.0 * md->itsSimulationElapsedTime)) % 2) ? true : false;
			if (md->itsSimulationElapsedTime >= 1.4)
			{
				SimulationEnd(md);
				md->itsFlashFlag = false;
			}
			break;

		default:
			break;
	}
}


unsigned int GetNum2DMazeBackgroundTextureRepetitions(void)
{
	return NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_MAZE;
}

void Get2DMazeKleinAxisColors(
	float	someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	someKleinAxisColors[0][0] = 1.00;
	someKleinAxisColors[0][1] = 0.00;
	someKleinAxisColors[0][2] = 0.00;
	someKleinAxisColors[0][3] = 1.00;

	someKleinAxisColors[1][0] = 0.00;
	someKleinAxisColors[1][1] = 0.50;
	someKleinAxisColors[1][2] = 0.50;
	someKleinAxisColors[1][3] = 1.00;
}


unsigned int GetNum2DMazeSprites(void)
{
	return 3;	//	maze, mouse and cheese
}

void Get2DMazeSpritePlacements(
	ModelData		*md,				//	input
	unsigned int	aNumSprites,		//	input
	Placement2D		*aPlacementBuffer)	//	output;  big enough to hold aNumSprites Placement2D's
{
	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game2DMaze,
		"Game2DMaze must be active");

	GEOMETRY_GAMES_ASSERT(
		aNumSprites == GetNum2DMazeSprites(),
		"Internal error:  wrong buffer size");

	aPlacementBuffer[0] = (Placement2D) {0.0, 0.0, false, 0.0, 1.0, 1.0};	//	the maze itself
	aPlacementBuffer[1] = md->itsGameOf.Maze2D.itsMousePlacement;
	aPlacementBuffer[2] = md->itsGameOf.Maze2D.itsCheesePlacement;
}
